﻿using System;
using System.Collections.Generic;

//Some namespace mappings
using D3D11 = SharpDX.Direct3D11;
using DXGI = SharpDX.DXGI;
using D3D = SharpDX.Direct3D;
using D3dDevice = SharpDX.Direct3D11.Device;

namespace Testing.GraphicsAdapterOverview.Graphics
{
    // Overview Feature levels:
    //http://msdn.microsoft.com/en-us/library/windows/desktop/ff476876(v=vs.85).aspx

    // Informations on WARP
    //http://msdn.microsoft.com/en-us/library/windows/desktop/gg615082(v=vs.85).aspx#capabilities

    /// <summary>
    /// All initialization logic for the D3D11 device
    /// </summary>
    public class DeviceHandlerD3D11
    {
        // Resources from Direct3D11 api
        private DXGI.Adapter1 m_dxgiAdapter;
        private D3dDevice m_device;
        private D3D11.DeviceContext m_immediateContext;

        // Parameters of created device
        private D3D11.DeviceCreationFlags m_creationFlags;
        private D3D.FeatureLevel m_featureLevel;

        /// <summary>
        /// Initializes a new instance of the <see cref="DeviceHandlerD3D11"/> class.
        /// </summary>
        /// <param name="dxgiHandler">The dxgi handler.</param>
        internal DeviceHandlerD3D11(DXGI.Adapter1 dxgiAdapter, bool debugEnabled)
        {
            m_dxgiAdapter = dxgiAdapter;

            //Define possible create flags
            D3D11.DeviceCreationFlags createFlagsBgra = D3D11.DeviceCreationFlags.BgraSupport;
            D3D11.DeviceCreationFlags createFlags = D3D11.DeviceCreationFlags.None;
            if (debugEnabled)
            {
                createFlagsBgra |= D3D11.DeviceCreationFlags.Debug;
                createFlags |= D3D11.DeviceCreationFlags.Debug;
            }

            //Define all steps on which we try to initialize Direct3D
            List<Tuple<D3D.FeatureLevel, D3D11.DeviceCreationFlags>> initParameterQueue =
                new List<Tuple<D3D.FeatureLevel, D3D11.DeviceCreationFlags>>();
            initParameterQueue.Add(Tuple.Create(
                D3D.FeatureLevel.Level_11_0, createFlagsBgra));
            initParameterQueue.Add(Tuple.Create(
                D3D.FeatureLevel.Level_10_1, createFlagsBgra));
            initParameterQueue.Add(Tuple.Create(
                D3D.FeatureLevel.Level_10_0, createFlagsBgra));
            initParameterQueue.Add(Tuple.Create(
                D3D.FeatureLevel.Level_10_0, createFlags));

            //Try to create the device, each defined configuration step by step
            foreach (Tuple<D3D.FeatureLevel, D3D11.DeviceCreationFlags> actInitParameters in initParameterQueue)
            {
                D3D.FeatureLevel featureLevel = actInitParameters.Item1;
                D3D11.DeviceCreationFlags direct3D11Flags = actInitParameters.Item2;

                try
                {
                    // Try to create the device using current parameters
                    using (D3D11.Device device = new D3D11.Device(dxgiAdapter, direct3D11Flags, featureLevel))
                    {
                        m_device = device.QueryInterface<D3dDevice>();
                    }

                    //Device successfully created, save all parameters and break this loop
                    m_featureLevel = featureLevel;
                    m_creationFlags = direct3D11Flags;
                    break;
                }
                catch (Exception) { }
            }

            //Throw exception on failure
            if (m_device == null) 
            { 
                throw new ApplicationException("Unable to initialize d3d11 device!"); 
            }

            //Get immediate context from the device
            m_immediateContext = m_device.ImmediateContext;
        }

        /// <summary>
        /// Unloads all resources.
        /// </summary>
        public void UnloadResources()
        {
            m_immediateContext = GraphicsHelper.DisposeObject(m_immediateContext);
            m_device = GraphicsHelper.DisposeObject(m_device);

            m_creationFlags = D3D11.DeviceCreationFlags.None;
            m_featureLevel = D3D.FeatureLevel.Level_11_0;
        }

        /// <summary>
        /// Gets current feature level.
        /// </summary>
        internal D3D.FeatureLevel FeatureLevel
        {
            get { return m_featureLevel; }
        }

        /// <summary>
        /// Gets the Direct3D 11 device.
        /// </summary>
        internal D3dDevice Device
        {
            get { return m_device; }
        }

        /// <summary>
        /// Gets the native pointer to the device object.
        /// </summary>
        public IntPtr DeviceNativePointer
        {
            get { return m_device.NativePointer; }
        }

        /// <summary>
        /// Gets the immediate context.
        /// </summary>
        internal D3D11.DeviceContext ImmediateContext
        {
            get { return m_immediateContext; }
        }

        /// <summary>
        /// Is device successfully initialized?
        /// </summary>
        public bool IsInitialized
        {
            get { return m_device != null; }
        }

        /// <summary>
        /// Gets a short description containing info about the created device.
        /// </summary>
        public string DeviceModeDescription
        {
            get
            {
                if (m_device == null) { return "None"; }

                return m_dxgiAdapter.Description1.Description + " - " + m_featureLevel + (this.IsDirect2DTextureEnabled ? " - Bgra" : " - No Bgra");
            }
        }

        /// <summary>
        /// Are Direct2D textures possible?
        /// </summary>
        public bool IsDirect2DTextureEnabled
        {
            get { return (m_creationFlags & D3D11.DeviceCreationFlags.BgraSupport) == D3D11.DeviceCreationFlags.BgraSupport; }
        }
    }
}
