﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

// Some namespace mappings
using D3D11 = SharpDX.Direct3D11;
using D3D10 = SharpDX.Direct3D10;
using D3D9 = SharpDX.Direct3D9;
using D3D = SharpDX.Direct3D;
using DXGI = SharpDX.DXGI;

namespace Testing.GraphicsAdapterOverview.Graphics
{
    public class EngineDevice
    {
        private const string CATEGORY_ADAPTER = "Adapter";

        private DXGI.Adapter1 m_adapter;
        private List<EngineOutputInfo> m_outputs;
        private bool m_isSoftwareAdapter;
        private bool m_debugEnabled;

        // Default information
        private Exception m_initializationException;

        // Handlers for different DirectX Apis
        private DeviceHandlerDXGI m_handlerDXGI;
        private DeviceHandlerD3D11 m_handlerD3D11;
        private DeviceHandlerD3D10 m_handlerD3D10;
        private DeviceHandlerD3D9 m_handlerD3D9;
        private DeviceHandlerD2D m_handlerD2D;
        private DeviceHandlerDWrite m_handlerDWrite;
        private DeviceHandlerWIC m_handlerWIC;

        // Possible antialiasing modes
        private DXGI.SampleDescription m_sampleDescWithAntialiasing;

        // Members for feature support information
        private bool m_isStandardAntialiasingSupported;

        /// <summary>
        /// Initializes a new instance of the <see cref="EngineDevice"/> class.
        /// </summary>
        /// <param name="hostAdapter">The host adapter.</param>
        private EngineDevice(DXGI.Adapter1 adapter, bool isSoftwareAdapter, bool debugEnabled)
        {
            m_adapter = adapter;
            m_isSoftwareAdapter = isSoftwareAdapter;
            m_debugEnabled = debugEnabled;

            // Set default antialiasing configurations
            m_sampleDescWithAntialiasing = new DXGI.SampleDescription(1, 0);

            // Create all output informations
            DXGI.Output[] outputs = m_adapter.Outputs;
            m_outputs = new List<EngineOutputInfo>(outputs.Length);
            for (int loop = 0; loop < outputs.Length; loop++)
            {
                m_outputs.Add(new EngineOutputInfo(loop, outputs[loop]));
            }

            // Initialize all direct3D APIs
            try
            {
                m_handlerD3D11 = new DeviceHandlerD3D11(adapter, debugEnabled);
                m_handlerDXGI = new DeviceHandlerDXGI(adapter, m_handlerD3D11.Device);
                m_handlerD3D10 = new DeviceHandlerD3D10(adapter, debugEnabled);
                m_handlerD3D9 = new DeviceHandlerD3D9(adapter, isSoftwareAdapter, debugEnabled);
                m_handlerD2D = new DeviceHandlerD2D(debugEnabled);
                m_handlerDWrite = new DeviceHandlerDWrite();
                m_handlerWIC = new DeviceHandlerWIC();
            }
            catch (Exception ex)
            {
                m_initializationException = ex;

                m_handlerD3D11 = null;
                m_handlerDXGI = null;
                m_handlerD3D10 = null;
                m_handlerD3D9 = null;
            }

            // Initialize handlers for feature support information
            if (m_initializationException == null)
            {
                m_isStandardAntialiasingSupported = CheckIsStandardAntialiasingPossible();
            }
        }

        /// <summary>
        /// Creates a list containing all available devices.
        /// </summary>
        public static IEnumerable<EngineDevice> CreateDevices(bool debugEnabled)
        {
            using(DXGI.Factory1 dxgiFactory = new DXGI.Factory1())
            {
                int adapterCount = dxgiFactory.GetAdapterCount1();
                for(int loop=0 ; loop<adapterCount; loop++)
                {
                    DXGI.Adapter1 actAdapter = dxgiFactory.GetAdapter1(loop);
                    DXGI.AdapterDescription1 actAdapterDescription = actAdapter.Description1;

                    // Check for software device (don't add software devices here, this is done one step later)
                    bool isSoftware =
                        (actAdapterDescription.Description == "Microsoft Basic Render Driver") ||
                        ((!string.IsNullOrEmpty(actAdapterDescription.Description)) && actAdapterDescription.Description.Contains("Software"));
                    if (!isSoftware)
                    {
                        // Create the device object
                        yield return new EngineDevice(actAdapter, isSoftware, debugEnabled);
                    }
                }

                // Try to append warp adapter (=> Software renderer)
                using (D3D11.Device warpDevice = new D3D11.Device(D3D.DriverType.Warp))
                using (DXGI.Device1 dxgiDevice = warpDevice.QueryInterface<DXGI.Device1>())
                {
                    DXGI.Adapter1 warpAdapter = dxgiDevice.GetParent<DXGI.Adapter1>();
                    if (warpAdapter != null)
                    {
                        yield return new EngineDevice(warpAdapter, true, debugEnabled);
                    }
                }
            }
        }

        /// <summary>
        /// Get the sample description for the given quality level.
        /// </summary>
        /// <param name="qualityLevel">The quality level for which a sample description is needed.</param>
        internal DXGI.SampleDescription GetSampleDescription(bool antialiasingEnabled)
        {
            if (antialiasingEnabled) { return m_sampleDescWithAntialiasing; }
            else { return new DXGI.SampleDescription(1, 0); }
        }

        /// <summary>
        /// Returns a <see cref="System.String" /> that represents this instance.
        /// </summary>
        /// <returns>
        /// A <see cref="System.String" /> that represents this instance.
        /// </returns>
        public override string ToString()
        {
            if (m_initializationException != null) { return m_adapter.Description1.Description; }
            else { return m_handlerD3D11.DeviceModeDescription; }
        }

        /// <summary>
        /// Checks for standard antialiasing support.
        /// </summary>
        private bool CheckIsStandardAntialiasingPossible()
        {
            // Check for quality levels
            int qualityLevels = m_handlerD3D11.Device.CheckMultisampleQualityLevels(GraphicsHelper.DEFAULT_TEXTURE_FORMAT, 4);

            // Generate sample descriptions for each possible quality level
            if(qualityLevels > 0)
            {
                m_sampleDescWithAntialiasing = new DXGI.SampleDescription(4, qualityLevels - 1);
            }

            return qualityLevels > 0;
        }

        /// <summary>
        /// Gets a list containing all outputs that belong to this device.
        /// </summary>
        [Browsable(false)]
        public List<EngineOutputInfo> Outputs
        {
            get { return m_outputs; }
        }

        /// <summary>
        /// Gets the description of this adapter.
        /// </summary>
        [Category(CATEGORY_ADAPTER)]
        public string AdapterDescription
        {
            get { return m_adapter.Description1.Description; }
        }

        /// <summary>
        /// Is this device loaded successfully.
        /// </summary>
        [Category(CATEGORY_ADAPTER)]
        public bool IsLoadedSuccessfully
        {
            get { return m_initializationException == null; }
        }

        /// <summary>
        /// Is this adapter a software adapter?
        /// </summary>
        [Category(CATEGORY_ADAPTER)]
        public bool IsSoftware
        {
            get { return m_isSoftwareAdapter; }
        }

        /// <summary>
        /// Checks for standard antialiasing support.
        /// </summary>
        [Category(CATEGORY_ADAPTER)]
        public bool IsStandardAntialiasingPossible
        {
            get 
            {
                return m_isStandardAntialiasingSupported; 
            }
        }

        [Category(CATEGORY_ADAPTER)]
        [DisplayName("Is Direct2D Texture possible")]
        public bool IsDirect2DTexturePossible
        { 
            get
            {
                if (m_handlerD3D11 == null) { return false; }
                else { return m_handlerD3D11.IsDirect2DTextureEnabled; }
            }
        }

        /// <summary>
        /// Gets the exception occurred during initialization of the driver (if any).
        /// </summary>
        [Browsable(false)]
        public Exception InitializationException
        {
            get { return m_initializationException; }
        }

        /// <summary>
        /// Gets the Direct3D 11 feature level
        /// </summary>
        [Category(CATEGORY_ADAPTER)]
        [DisplayName("Direct3D 11 Feature Level")]
        public string Direct3D11FeatureLevel
        {
            get
            {
                if (m_handlerD3D11 == null) { return string.Empty; }
                else { return m_handlerD3D11.FeatureLevel.ToString(); }
            }
        }
    }
}
